#include <stdio.h>
#include <stdlib.h>
#include <win32/windows.h>
#include <io.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <math.h>

#define EARTH_RADIUS 6371.0

#define LINE_LENGTH 1040
#define LOOK_AHEAD_LINES 5
#define MAX_OUTPUT_HEIGHT 3000

#define UNDEF_ANGLE 9999.0

int map_width;
int map_height;
double west, east, south, north;

const double M_PI = 3.14159265358979323846;

void die(const char*why)
{
    printf(why);
    exit(-1);
}

typedef struct {
    double latitude;
    double longitude;
    double altitude;
} POLAR_POS;

double rad_to_deg(double angle)
{
        return angle*180/M_PI;
}

typedef struct {
    double x;
    double y;
    double z;
} KAR_POS;

void save_bitmap(unsigned char *info, int line_length, int num_of_lines, const char *file_name)
{
    int handle = 0;
    int i;

    BITMAPFILEHEADER file_header = {
        0x4D42,
        sizeof(BITMAPFILEHEADER)+sizeof(BITMAPINFOHEADER)+256*sizeof(RGBQUAD) + num_of_lines * line_length,
        0,
        0,
        sizeof(BITMAPFILEHEADER)+sizeof(BITMAPINFOHEADER)+256*sizeof(RGBQUAD)
    };
    BITMAPINFOHEADER info_header = {
        sizeof(BITMAPINFOHEADER),       // size
        line_length,                    // width
        num_of_lines,                   // height
        1,                              // planes = 1
        8,                              // bpp
        BI_RGB,                         // compression
        0,                              // size, may be 0
        100,                            // hor resolution
        100,                            // ver resolution
        0,                              // colors used in color table,
        0                               // important colors
    };

    handle = open(file_name, O_RDWR|O_CREAT|O_TRUNC|O_BINARY,S_IREAD|S_IWRITE);
    if (handle == -1) {
        printf("Error opening output file\n");
        exit(-1);
    }

    write(handle, &file_header, sizeof(BITMAPFILEHEADER));
    write(handle, &info_header, sizeof(BITMAPINFOHEADER));
    for (i=0; i<256; i++) {
        RGBQUAD quad = {i,i,i,0};
        write(handle, &quad, sizeof(RGBQUAD));
    }

    write(handle, info, line_length * num_of_lines);

    close(handle);
}

POLAR_POS input_file[MAX_OUTPUT_HEIGHT];
KAR_POS current_positions[MAX_OUTPUT_HEIGHT];
KAR_POS normals[MAX_OUTPUT_HEIGHT];
int input_file_length;

void read_file(const char *file_name)
{
    FILE *handle;
    char line[300];

    input_file_length = 0;

    handle = fopen(file_name, "r");
    if (handle == NULL) {
        printf("Error opening input file\n");
        exit(-1);
    }

    while (fgets(line, 299, handle) != NULL) {
        input_file[input_file_length].latitude =
                atof(line+25)/180*M_PI;
        input_file[input_file_length].longitude =
                atof(line+37)/180*M_PI;
		input_file[input_file_length].altitude =
                atof(line+48);

        input_file_length ++;
    }

    fclose(handle);
}

int get_num_of_lines(void)
{
    return input_file_length-LOOK_AHEAD_LINES;
}

// -----------------------------------------------------------------------------------------
// calculations
// -----------------------------------------------------------------------------------------

KAR_POS polar_to_kar(const POLAR_POS *input)
{
    KAR_POS result;

    result.x = cos(input->latitude)*cos(input->longitude);
    result.y = cos(input->latitude)*sin(input->longitude);
    result.z = sin(input->latitude);

    return result;
}

POLAR_POS kar_to_polar(const KAR_POS *input)
{
    POLAR_POS result;

    result.latitude = asin(input->z);
    result.longitude = asin(input->y / cos(result.latitude));

    return result;
}

KAR_POS vector_mul(const KAR_POS *a, const KAR_POS *b)
{
    KAR_POS result;

    result.x = a->y*b->z - a->z*b->y;
    result.y = a->z*b->x - a->x*b->z;
    result.z = a->x*b->y - a->y*b->x;

    return result;
}

void normalize(KAR_POS *a)
{
    double norm = sqrt(a->x*a->x + a->y*a->y + a->z*a->z);

    a->x /= norm;
    a->y /= norm;
    a->z /= norm;
}

KAR_POS combine(const KAR_POS *main, const KAR_POS *normal, double angle)
{
    KAR_POS result;

    result.x = cos(angle)*main->x + sin(angle)*normal->x;
    result.y = cos(angle)*main->y + sin(angle)*normal->y;
    result.z = cos(angle)*main->z + sin(angle)*normal->z;

    return result;
}

unsigned char *map;
double angles[LINE_LENGTH];
double sat_angles[LINE_LENGTH];

void pre_calculate()
{
    int x, y;

    for(y=0; y<get_num_of_lines(); y++) {
        KAR_POS future_pos;
        KAR_POS normal;
        KAR_POS current_pos = polar_to_kar(&input_file[y]);
        POLAR_POS future_pos_polar = input_file[y+LOOK_AHEAD_LINES];

        // the future pos has to be corrected to remove the effect of earth's rotation
        future_pos_polar.longitude += (double)LOOK_AHEAD_LINES*2*M_PI/24/7200;

        future_pos = polar_to_kar(&future_pos_polar);

        normal = vector_mul(&current_pos, &future_pos);
                normalize(&normal);

        normals[y] = normal;
        current_positions[y] = current_pos;
    }

    // Calculate the sat's viewing angles for each pixel
    for(x=0; x<LINE_LENGTH; x++) {
        sat_angles[x] = UNDEF_ANGLE;    // so we can filter that afterwards
    }

    for(x=0; x<79; x++) {
        sat_angles[499+x] = 16.98 * (double)x / 79;
        sat_angles[499-x] = -sat_angles[499+x];
    }
    for(x=79; x<189; x++) {
        sat_angles[499+x] = 16.98 + 17.85 * (double)(x-79) / 110;
        sat_angles[499-x] = -sat_angles[499+x];
    }
    for(x=189; x<272; x++) {
        sat_angles[499+x] = 34.83 + 9.00 * (double)(x-189) / 83;
        sat_angles[499-x] = -sat_angles[499+x];
    }
    for(x=272; x<334; x++) {
        sat_angles[499+x] = 43.83 + 5.01 * (double)(x-272) / 62;
        sat_angles[499-x] = -sat_angles[499+x];
    }
    for(x=334; x<455; x++) {
        sat_angles[499+x] = 48.84 + 6.56 * (double)(x-334) / 121;
        sat_angles[499-x] = -sat_angles[499+x];
    }

    // convert to radians
    for(x=0; x<LINE_LENGTH; x++) {
        if (sat_angles[x] != UNDEF_ANGLE) {
            sat_angles[x] = sat_angles[x] / 180 * M_PI;
        }
    }
}

POLAR_POS get_pos(int x, int y)
{
    KAR_POS pos;

    pos = combine(&current_positions[y], &normals[y], angles[x]);

    return kar_to_polar(&pos);
}

void draw_bitmap(const char *file_name)
{
    int x,y;
    unsigned char *picture;

    picture = (unsigned char*)malloc(LINE_LENGTH * get_num_of_lines());
    memset(picture, 0, LINE_LENGTH * get_num_of_lines());

    for(y = 0; y < get_num_of_lines(); y ++) {
        printf("Processing line %d\n", y);

		// Prepare line information
		for(x=0; x<LINE_LENGTH; x++) {
            if (sat_angles[x] == UNDEF_ANGLE) {
                angles[x] = UNDEF_ANGLE;
            } else {
                angles[x] = sat_angles[x] - asin(sin(sat_angles[x]) * (1 + (double)input_file[y].altitude/EARTH_RADIUS));
            }
		}

		for(x=0; x<LINE_LENGTH; x++) {
            if (angles[x] != UNDEF_ANGLE) {
                POLAR_POS xy = get_pos(x,y);

#ifdef DRAW_LAT_LONG
                POLAR_POS xy1 = get_pos(x,y+1);
                POLAR_POS x1y = get_pos(x+1,y);
                POLAR_POS xy_1 = get_pos(x,y-1);
                POLAR_POS x_1y = get_pos(x-1,y);

                // draw the meridians and parallels
                if ((floor(rad_to_deg(xy.latitude)) < floor(rad_to_deg(xy1.latitude))) ||
                        (floor(rad_to_deg(xy.latitude)) < floor(rad_to_deg(x1y.latitude))) ||
                        (floor(rad_to_deg(xy.latitude)) < floor(rad_to_deg(xy_1.latitude))) ||
                        (floor(rad_to_deg(xy.latitude)) < floor(rad_to_deg(x_1y.latitude)))) {
                    picture[y * LINE_LENGTH + x] = 100;
                }
                if ((floor(rad_to_deg(xy.longitude)) < floor(rad_to_deg(x1y.longitude))) ||
                        (floor(rad_to_deg(xy.longitude)) < floor(rad_to_deg(xy1.longitude))) ||
                        (floor(rad_to_deg(xy.longitude)) < floor(rad_to_deg(x_1y.longitude))) ||
                        (floor(rad_to_deg(xy.longitude)) < floor(rad_to_deg(xy_1.longitude)))) {
                    picture[y * LINE_LENGTH + x] = 100;
                }
                if ((floor(rad_to_deg(xy.latitude)/10) < floor(rad_to_deg(xy1.latitude)/10)) ||
                        (floor(rad_to_deg(xy.latitude)/10) < floor(rad_to_deg(x1y.latitude)/10)) ||
                        (floor(rad_to_deg(xy.latitude)/10) < floor(rad_to_deg(xy_1.latitude)/10)) ||
                        (floor(rad_to_deg(xy.latitude)/10) < floor(rad_to_deg(x_1y.latitude)/10))) {
                    picture[y * LINE_LENGTH + x] = 0;
                }
                if ((floor(rad_to_deg(xy.longitude)/10) < floor(rad_to_deg(x1y.longitude)/10)) ||
                        (floor(rad_to_deg(xy.longitude())/10) < floor(rad_to_deg(xy1.longitude)/10)) ||
                        (floor(rad_to_deg(xy.longitude())/10) < floor(rad_to_deg(x_1y.longitude)/10)) ||
                        (floor(rad_to_deg(xy.longitude())/10) < floor(rad_to_deg(xy_1.longitude)/10))) {
                    picture[y * LINE_LENGTH + x] = 0;
                }
#endif                                  // DRAW_LAT_LONG

                // draw the map
                int map_x = (int)floor((double)(xy.longitude-west)/(east-west)*map_width);
                int map_y = (int)floor((double)(xy.latitude-south)/(north-south)*map_height);
                if ( (map_x>=0) && (map_x<map_width) && (map_y>=0) && (map_y<map_height)) {
                    if (map[map_y * map_width + map_x] == 0) {
                        picture[y * LINE_LENGTH + x] = 255;
                    }
                }
            }
        }

#ifdef DRAW_CENTER_LINE
        // draw the center line
        picture[y * LINE_LENGTH + LINE_LENGTH/2] = 255;
#endif

		if (y % 120 <= 1) {
            for(x=39; x<86; x++) {
                picture[y * LINE_LENGTH + LINE_LENGTH - 1 - x] = 0;
			}
        } else if ( y%120 <= 3) {
            for(x=39; x<86; x++) {
                picture[y * LINE_LENGTH + LINE_LENGTH - 1 - x] = 255;
			}
        } else {
            for(x=39; x<86; x++) {
                picture[y * LINE_LENGTH + LINE_LENGTH - 1 - x] = 128;
			}
        }

        // Two vertical lines on each side
        picture[y * LINE_LENGTH + LINE_LENGTH - 86] = 255;
        picture[y * LINE_LENGTH + 44] = 255;
    }

    save_bitmap(picture, LINE_LENGTH, get_num_of_lines(), file_name);
}


void read_map()
{
    BITMAPFILEHEADER file_header;
    BITMAPINFOHEADER info_header;
    int hnd;

    hnd = open("map.bmp", O_RDONLY|O_BINARY);
    if (hnd == -1) {
        printf("Cannot open input file\n");
        return;
    }

    read(hnd, &file_header, sizeof(BITMAPFILEHEADER));
    if (file_header.bfType != 0x4D42) {
        die("Wrong MAP file\n");
    }

    read(hnd, &info_header, sizeof(BITMAPINFOHEADER));
    if (info_header.biCompression != BI_RGB) {
        die("MAP must be uncompressed\n");
    }
    if (info_header.biBitCount != 8) {
        die("MAP must have 8bpp\n");
    }
    map_width = info_header.biWidth;
    map_height = info_header.biHeight;

    lseek(hnd, file_header.bfOffBits, SEEK_SET);

    map = (unsigned char *)malloc(map_width * map_height);
    read(hnd, map, map_width * map_height);

    close(hnd);
}

void read_map_cfg()
{
    FILE *cfg_file;

    cfg_file = fopen("map.cfg", "r");
    fscanf(cfg_file, "%lf %lf %lf %lf", &west,&east, &south, &north);
    fclose(cfg_file);
}

int main(int argc, char *argv[])
{
    const char *input_file_name, *output_file_name;

    if (argc < 2) {
        printf("Usage:\n");
        printf("\tTRACKER <input-track-file> [<output-bitmap>]\n");
        return 0;
    }

    input_file_name = argv[1];
    if (argc > 2) {
        output_file_name = argv[2];
    } else {
        output_file_name = "OUTPUT.BMP";
    }

    printf("Reading map file...\n");
    read_map();
    printf("Reading map cfg file...\n");
    read_map_cfg();

    printf("Reading input track file...\n");
    read_file(input_file_name);

    printf("Performing precalculations...\n");
    pre_calculate();

    draw_bitmap(output_file_name);

    return 0;
}
